<?php
namespace Codexpert\CoDesigner_Pro;

use Elementor\Repeater;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Background;
use Codexpert\CoDesigner\App\Controls\Group_Control_Gradient_Text;

class Shipping_Address extends Widget_Base {

	public $id;

	public function __construct( $data = [], $args = null ) {
	    parent::__construct( $data, $args );

	    $this->id = wcd_get_widget_id( __CLASS__ );
	    $this->widget = wcd_get_widget( $this->id );
	}

	public function get_script_depends() {
		return [];
	}

	public function get_style_depends() {
		return [];
	}

	public function get_name() {
		return $this->id;
	}

	public function get_title() {
		return $this->widget['title'];
	}

	public function get_icon() {
		return $this->widget['icon'];
	}

	public function get_categories() {
		return $this->widget['categories'];
	}

	protected function register_controls() {

		$this->start_controls_section(
			'shipping_title',
			[
				'label' => __( 'Section Title', 'codesigner-pro' ),
				'tab' 	=> Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
            'shipping_title_show',
            [
                'label'         => __( 'Show/Hide Title', 'codesigner-pro' ),
                'type'          => Controls_Manager::SWITCHER,
                'label_on'      => __( 'Show', 'codesigner-pro' ),
                'label_off'     => __( 'Hide', 'codesigner-pro' ),
                'return_value'  => 'yes',
                'default'       => 'yes',
            ]
        );
		$this->add_control(
		    'shipping_title_text',
		    [
		        'label' 		=> __( 'Text', 'codesigner-pro' ),
		        'type' 			=> Controls_Manager::TEXT,
		        'default' 		=> __( 'Shipping Address', 'codesigner-pro' ),
                'condition' 	=> [
                    'shipping_title_show' => 'yes'
                ],
		        'dynamic' 		=> [
		            'active' 		=> true,
		        ]
		    ]
		);

		$this->add_control(
			'payment_title_tag',
			[
				'label' 	=> __( 'HTML Tag', 'codesigner-pro' ),
				'type' 		=> Controls_Manager::SELECT,
				'default' 	=> 'h3',
				'options' 	=> [
					'h1'  => __( 'H1', 'codesigner-pro' ),
					'h2'  => __( 'H2', 'codesigner-pro' ),
					'h3'  => __( 'H3', 'codesigner-pro' ),
					'h4'  => __( 'H4', 'codesigner-pro' ),
					'h5'  => __( 'H5', 'codesigner-pro' ),
					'h6'  => __( 'H6', 'codesigner-pro' ),
				],
                'condition' => [
                    'shipping_title_show' => 'yes'
                ],
			]
		);

		$this->add_control(
            'shipping_title_alignment',
            [
                'label' 	   => __( 'Alignment', 'codesigner-pro' ),
                'type' 		   => Controls_Manager::CHOOSE,
                'options' 	   => [
                    'left' 		=> [
                        'title' 	=> __( 'Left', 'codesigner-pro' ),
                        'icon' 		=> 'eicon-text-align-left',
                    ],
                    'center' 	=> [
                        'title' 	=> __( 'Center', 'codesigner-pro' ),
                        'icon' 		=> 'eicon-text-align-center',
                    ],
                    'right' 	=> [
                        'title' 	=> __( 'Right', 'codesigner-pro' ),
                        'icon' 		=> 'eicon-text-align-right',
                    ],
                ],
                'default' 	=> 'left',
                'toggle' 	=> true,
                'condition' => [
                    'shipping_title_show' => 'yes'
                ],
                'selectors' => [
                    '{{WRAPPER}} .wl-shipping-title' => 'text-align: {{VALUE}};',
                ],
            ]
        );

		$this->end_controls_section();

		$this->start_controls_section(
			'content_section',
			[
				'label' => __( 'Shipping Fields', 'codesigner-pro' ),
				'tab' 	=> Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'default_checked',
			[
				'label'	 	=> __( 'Default Checked?', 'codesigner-pro' ),
				'type' 		=> Controls_Manager::SWITCHER,
				'label_on' 	=> __( 'Yes', 'codesigner-pro' ),
				'label_off' => __( 'No', 'codesigner-pro' ),
				'return_value' 	=> 'yes',
				'default' 		=> '',
			]
		);

        $this->add_control(
            'shipping_toggle_caption',
            [
                'label' 		=> __( 'Toggle Caption', 'codesigner-pro' ),
                'type' 			=> Controls_Manager::TEXT,
                'default' 		=> __( 'Ship to a different address?', 'codesigner-pro' ),
                'separator' 	=> 'after',
                'dynamic' 		=> [
                    'active' 		=> true,
                ]
            ]
        );

		$repeater = new Repeater();

		$repeater->add_control(
			'shipping_input_label', [
				'label' 	=> __( 'Input Label', 'codesigner-pro' ),
				'type' 		=> Controls_Manager::TEXT,
				'default' 	=> __( 'New Section' , 'codesigner-pro' ),
				'label_block' 	=> true,
				'separator' 	=> 'after',
			]
		);

		$repeater->add_control(
			'shipping_input_class', [
				'label' 	=> __( 'Class Name', 'codesigner-pro' ),
				'type' 		=> Controls_Manager::SELECT,
				'default' 	=> 'form-row-wide',
				'options' 	=> [
					'form-row-first' 	=> 'form-row-first',
					'form-row-last' 	=> 'form-row-last',
					'form-row-wide' 	=> 'form-row-wide',
				],
			]
		);

		$repeater->add_control(
			'shipping_input_type', [
				'label' 	=> __( 'Input Type', 'codesigner-pro' ),
				'type' 		=> Controls_Manager::SELECT2,
				'default' 	=> 'text',
				'options' 	=> [
					'textarea'			=> __( 'Textarea', 'codesigner-pro' ),
					'checkbox'			=> __( 'Checkbox', 'codesigner-pro' ),
					'text'				=> __( 'Text', 'codesigner-pro' ),
					'password'			=> __( 'Password', 'codesigner-pro' ),
					'date'				=> __( 'Date', 'codesigner-pro' ),
					'number'			=> __( 'Number', 'codesigner-pro' ),
					'email'				=> __( 'Email', 'codesigner-pro' ),
					'url'				=> __( 'Url', 'codesigner-pro' ),
					'tel'				=> __( 'Tel', 'codesigner-pro' ),
					'select'			=> __( 'Select', 'codesigner-pro' ),
					'radio'				=> __( 'Radio', 'codesigner-pro' ),
				],
			]
		);

		$repeater->add_control(
			'shipping_input_options', [
				'label' 	=> __( 'Options', 'codesigner-pro' ),
				'type' 		=> Controls_Manager::TEXTAREA,
				'default' 	=> implode( PHP_EOL, [ __( 'Option 1', 'codesigner-pro' ), __( 'Option 2', 'codesigner-pro' ), __( 'Option 3', 'codesigner-pro' ) ] ),
				'label_block' 	=> true,
				'conditions' 	=> [
					'relation' 	=> 'or',
					'terms' 	=> [
						[
							'name' 		=> 'shipping_input_type',
							'operator' 	=> '==',
							'value' 	=> 'select',
						],
						[
							'name' 		=> 'shipping_input_type',
							'operator' 	=> '==',
							'value' 	=> 'radio',
						],
					],
				],
			]
		);

		$repeater->add_control(
			'shipping_input_name', [
				'label' 		=> __( 'Field Name', 'codesigner-pro' ),
				'type' 			=> Controls_Manager::TEXT,
				'default' 		=> 'name_' . rand( 111, 999 ),
				'label_block' 	=> true,
			]
		);

		$repeater->add_control(
			'shipping_input_placeholder', [
				'label' 		=> __( 'Placeholder', 'codesigner-pro' ),
				'type' 			=> Controls_Manager::TEXT,
				'default' 		=> __( 'Placeholder' , 'codesigner-pro' ),
				'label_block' 	=> true,
			]
		);

		$repeater->add_control(
			'shipping_input_autocomplete', [
				'label' 		=> __( 'Autocomplete Value', 'codesigner-pro' ),
				'type' 			=> Controls_Manager::TEXT,
				'default' 		=> __( 'Given value' , 'codesigner-pro' ),
				'label_block' 	=> true,
			]
		);

		$repeater->add_control(
			'shipping_show_in_mail', [
				'label' 		=> __( 'Show in Email', 'codesigner-pro' ),
				'type' 			=> Controls_Manager::SELECT,
				'default' 		=> 'disable',
				'options' 		=> [ 'disable' => __( 'Disable', 'codesigner-pro' ) ] + wcd_get_wc_email_hooks(),
				'label_block' 	=> true,
				'conditions' 	=> [
					'terms' => [
						[
							'name' => 'shipping_input_name',
							'operator' => '!in',
							'value' => wcd_wc_fields( 'shipping' ),
						],
					],
				],
			]
		);
		
		$repeater->add_control(
			'shipping_show_in_thankyou', [
				'label' 		=> __( 'Show on Thankyou Page', 'codesigner-pro' ),
				'type' 			=> Controls_Manager::SELECT,
				'default' 		=> 'disable',
				'options' 		=> [ 'disable' => __( 'Disable', 'codesigner-pro' ) ] + wcd_get_wc_thankyou_hooks(),
				'label_block' 	=> true,
				'conditions' 	=> [
					'terms' => [
						[
							'name' => 'shipping_input_name',
							'operator' => '!in',
							'value' => wcd_wc_fields( 'shipping' ),
						],
					],
				],
			]
		);

		$repeater->add_control(
			'shipping_show_in_editpage', [
				'label' 		=> __( 'Show on Order Edit Page', 'codesigner-pro' ),
				'type' 			=> Controls_Manager::SELECT,
				'default' 		=> 'disable',
				'options' 		=> [ 'disable' => __( 'Disable', 'codesigner-pro' ) ] + wcd_get_wc_edit_order_hooks(),
				'label_block' 	=> true,
				'conditions' 	=> [
					'terms' => [
						[
							'name' => 'shipping_input_name',
							'operator' => '!in',
							'value' => wcd_wc_fields( 'shipping' ),
						],
					],
				],
			]
		);

		$repeater->add_control(
			'shipping_input_required',
			[
				'label'         => __( 'Required', 'codesigner-pro' ),
				'type'          => Controls_Manager::SWITCHER,
				'label_on'      => __( 'yes', 'codesigner-pro' ),
				'label_off'     => __( 'no', 'codesigner-pro' ),
				'return_value'  => true,
				'default'       => true,
			]
		);

		$this->add_control(
			'shipping_form_items',
			[
				'label' 		=> __( '', 'codesigner-pro' ),
				'type' 			=> Controls_Manager::REPEATER,
				'fields' 		=> $repeater->get_controls(),
				'default' 		=> wcd_checkout_fields( 'shipping' ),
				'title_field' 	=> '{{{ shipping_input_label }}}',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'shipping_title_style',
			[
				'label' 	=> __( 'Title', 'codesigner-pro' ),
				'tab' 		=> Controls_Manager::TAB_STYLE,
                'condition' => [
                    'shipping_title_show' => 'yes'
                ],
			]
		);

		// start default style
		$this->add_control(
			'shipping_default_styles',
			[
				'label' 	=> __( 'Display', 'codesigner-pro' ),
				'type' 		=> Controls_Manager::HIDDEN,
				'selectors' => [
					'.wl {{WRAPPER}} .wl-shipping p label' 	=> 'display: block',
					'.wl {{WRAPPER}} .form-row-first' 		=> 'width: 48%;float: left;margin-right:0',
					'.wl {{WRAPPER}} .form-row-last' 		=> 'width: 48%;float: right;',
					'.wl {{WRAPPER}} .form-row-wide' 		=> 'clear: both;',
					'.wl {{WRAPPER}} .shipping-checkbox-area-hidden' 		=> 'display: none;',
					'.rtl.wl {{WRAPPER}} .form-row-first' 	=> 'width: 48%;float: right;margin-right: 0px;margin-left: 0px;',
					'.rtl.wl {{WRAPPER}} .wl-shipping .woocommerce-input-wrapper input[type="radio"]' => 'margin-left: 10px;margin-right: 1px;',

					'.wl {{WRAPPER}} .wl-shipping .woocommerce-input-wrapper input[type="text"],
					.wl {{WRAPPER}} .wl-shipping .woocommerce-input-wrapper input[type="password"],
					.wl {{WRAPPER}} .wl-shipping .woocommerce-input-wrapper input[type="datetime"],
					.wl {{WRAPPER}} .wl-shipping .woocommerce-input-wrapper input[type="datetime-local"],
					.wl {{WRAPPER}} .wl-shipping .woocommerce-input-wrapper input[type="date"],
					.wl {{WRAPPER}} .wl-shipping .woocommerce-input-wrapper input[type="month"],
					.wl {{WRAPPER}} .wl-shipping .woocommerce-input-wrapper input[type="time"],
					.wl {{WRAPPER}} .wl-shipping .woocommerce-input-wrapper input[type="week"],
					.wl {{WRAPPER}} .wl-shipping .woocommerce-input-wrapper input[type="number"],
					.wl {{WRAPPER}} .wl-shipping .woocommerce-input-wrapper input[type="email"],
					.wl {{WRAPPER}} .wl-shipping .woocommerce-input-wrapper input[type="url"],
					.wl {{WRAPPER}} .wl-shipping .woocommerce-input-wrapper input[type="tel"],
					.wl {{WRAPPER}} .wl-shipping .woocommerce-input-wrapper select' => 'width: 100%',
				],
				'default' => 'traditional',
			]
		);
		// end default css

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 		=> 'shipping_title_typographyrs',
				'label' 	=> __( 'Typography', 'codesigner-pro' ),
				'selector' 	=> '{{WRAPPER}} .wl-shipping-title',
				'fields_options' 	=> [
					'typography' 	=> [ 'default' => 'yes' ],
					'font_size' 	=> [ 'default' => [ 'size' => 22 ] ],
					// 'line_height' 	=> [ 'default' => [ 'size' => 37 ] ],
		            'font_family' 	=> [ 'default' => 'Montserrat' ],
		            'font_weight' 	=> [ 'default' => 500 ],
				],
			]
		);

		$this->add_group_control(
			Group_Control_Gradient_Text::get_type(),
			[
				'name' 		=> 'shipping_title_color',
				'selector' 	=> '{{WRAPPER}} .wl-shipping-title',
				'fields_options' => [
					'color' => [ 'default' => 'var(--wl-black)' ], 
				],
			]
		);

		$this->add_group_control(
			Group_Control_Background::get_type(),
			[
				'name' 		=> 'shipping_input_background_color',
				'label' 	=> __( 'Background', 'codesigner-pro' ),
				'types' 	=> [ 'classic', 'gradient' ],
				'selector' 	=> '{{WRAPPER}} .wl-shipping-title',
				'separator' => 'before',
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' 		=> 'shipping_title_border',
				'label' 	=> __( 'Border', 'codesigner-pro' ),
				'selector' 	=> '{{WRAPPER}} .wl-shipping-title',
				'separator' => 'before',
			]
		);

        $this->add_control(
			'shipping_title_border_radius',
			[
				'label' 	=> __( 'Border Radius', 'codesigner-pro' ),
				'type' 		=> Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .wl-shipping-title' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'shipping_title_padding',
			[
				'label' 	=> __( 'Padding', 'codesigner-pro' ),
				'type' 		=> Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .wl-shipping-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'shipping_title_margin',
			[
				'label'         => __( 'Margin', 'codesigner-pro' ),
				'type'          => Controls_Manager::DIMENSIONS,
				'size_units'    => [ 'px', '%' ],
				'selectors'     => [
					'{{WRAPPER}} .wl-shipping-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		/**
		 * Input Color
		 */
		$this->start_controls_section(
			'shipping_toggle',
			[
				'label' => __( 'Toggle Caption', 'codesigner-pro' ),
				'tab' 	=> Controls_Manager::TAB_STYLE,
			]
		);


		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 		=> 'shipping_toggle_typographyrs',
				'label' 	=> __( 'Typography', 'codesigner-pro' ),
				'selector' 	=> '{{WRAPPER}} .shipping-checkbox-caption',
				'fields_options' 	=> [
					'typography' 	=> [ 'default' => 'yes' ],
					'font_size' 	=> [ 'default' => [ 'size' => 14 ] ],
					// 'line_height' 	=> [ 'default' => [ 'size' => 37 ] ],
		            'font_family' 	=> [ 'default' => 'Montserrat' ],
		            'font_weight' 	=> [ 'default' => 400 ],
				],
			]
		);

		$this->add_control(
			'shipping_toggle_color',
			[
				'label'     => __( 'Text Color', 'codesigner-pro' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .shipping-checkbox-caption' => 'color: {{VALUE}}',
				],
			]
		);

		$this->end_controls_section();

		/**
		 * Input Label Color
		 */
		$this->start_controls_section(
			'shipping_style',
			[
				'label' => __( 'Labels', 'codesigner-pro' ),
				'tab' 	=> Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 		=> 'shipping_label_typographyrs',
				'label' 	=> __( 'Typography', 'codesigner-pro' ),
				'selector' 	=> '{{WRAPPER}} .wl-shipping label',
				'fields_options' 	=> [
					'typography' 	=> [ 'default' => 'yes' ],
					'font_size' 	=> [ 'default' => [ 'size' => 14 ] ],
					'line_height' 	=> [ 'default' => [ 'size' => 37 ] ],
		            'font_family' 	=> [ 'default' => 'Montserrat' ],
		            'font_weight' 	=> [ 'default' => 400 ],
				],
			]
		);


        $this->add_control(
			'shipping_label_color',
			[
				'label'     => __( 'Text Color', 'codesigner-pro' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .wl-shipping label' => 'color: {{VALUE}}',
				],
				'default' => 'var(--wl-black)'
			]
		);

        $this->add_control(
        	'shipping_label_padding',
        	[
        		'label' => __( 'Padding', 'codesigner-pro' ),
        		'type' 	=> Controls_Manager::DIMENSIONS,
        		'size_units' 	=> [ 'px', '%', 'em' ],
        		'selectors' 	=> [
        			'{{WRAPPER}} .wl-shipping label' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
        		],
        	]
        );

		$this->add_control(
			'shipping_label_line_hight',
			[
				'label' 		=> __( 'Line Height', 'codesigner-pro' ),
				'type' 			=> Controls_Manager::SLIDER,
				'size_units' 	=> [ 'px', '%' ],
				'range' => [
					'px' => [
						'min' 	=> 20,
						'max' 	=> 100,
						'step' 	=> 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' 	=> [
					'unit' 	=> 'px',
					'size' 	=> 25,
				],
				'selectors' => [
					'{{WRAPPER}} .wl-shipping label' => 'line-height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		/**
		 * Input Color
		 */
		$this->start_controls_section(
			'shipping_input_style',
			[
				'label' => __( 'Input Fields', 'codesigner-pro' ),
				'tab' 	=> Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' 		=> 'shipping_input_typographyrs',
				'label' 	=> __( 'Typography', 'codesigner-pro' ),
				'fields_options' 	=> [
					'typography' 	=> [ 'default' => 'yes' ],
					'font_size' 	=> [ 'default' => [ 'size' => 14 ] ],
					// 'line_height' 	=> [ 'default' => [ 'size' => 37 ] ],
		            'font_family' 	=> [ 'default' => 'Montserrat' ],
		            'font_weight' 	=> [ 'default' => 400 ],
				],
				'selector' 	=> '{{WRAPPER}} .wl-shipping input, 
								{{WRAPPER}} .wl-shipping select, 
								{{WRAPPER}} .wl-shipping option,
								{{WRAPPER}} .wl-shipping textarea',
			]
		);

		$this->add_control(
			'shipping_input_color',
			[
				'label'     => __( 'Input Text Color', 'codesigner-pro' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .wl-shipping input, 
					 {{WRAPPER}} .wl-shipping select, 
					 {{WRAPPER}} .wl-shipping option,
					 {{WRAPPER}} .wl-shipping textarea' => 'color: {{VALUE}}',
				],
				'default' => 'var(--wl-gray)'
			]
		);

		$this->add_control(
			'shipping_input_background_color',
			[
				'label'     => __( 'Background Color', 'codesigner-pro' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .wl-shipping input, 
					 {{WRAPPER}} .wl-shipping select, 
					 {{WRAPPER}} .wl-shipping option,
					 {{WRAPPER}} .wl-shipping .select2-container .select2-selection--single,
					 {{WRAPPER}} .wl-shipping textarea' => 'background-color: {{VALUE}}',
				],
			]
		);

        $this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' 		=> 'shipping_input_border',
				'label' 	=> __( 'Border', 'codesigner-pro' ),
				'separator' => 'before',
				'fields_options'	=> [
					'border' 		=> [
						'default' 	=> 'solid',
					],
					'width' 		=> [
						'default' 	=> [
							'top' 		=> '1',
							'right' 	=> '1',
							'bottom' 	=> '1',
							'left' 		=> '1',
							'isLinked' 	=> false,
						],
					],
					'color' => [
						'default' => 'var(--wl-light-border)',
					],
				],
				'selector' 	=> '
					{{WRAPPER}} .wl-shipping input, 
					{{WRAPPER}} .wl-shipping select,
					{{WRAPPER}} .wl-shipping .select2-container .select2-selection--single,
					{{WRAPPER}} .wl-shipping textarea',
			]
		);

        $this->add_control(
			'shipping_input_border_radius',
			[
				'label' => __( 'Border Radius', 'codesigner-pro' ),
				'type' 	=> Controls_Manager::DIMENSIONS,
				'size_units' 	=> [ 'px', '%', 'em' ],
				'selectors' 	=> [
					'{{WRAPPER}} .wl-shipping input, 
					 {{WRAPPER}} .wl-shipping select,
					 {{WRAPPER}} .wl-shipping textarea' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'shipping_input_padding',
			[
				'label' => __( 'Padding', 'codesigner-pro' ),
				'type' 	=> Controls_Manager::DIMENSIONS,
				'size_units' 	=> [ 'px', '%', 'em' ],
				'selectors' 	=> [
					'{{WRAPPER}} .wl-shipping input, 
					 {{WRAPPER}} .wl-shipping select,
					 {{WRAPPER}} .wl-shipping textarea' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

	}

	protected function render() {
		if ( ! function_exists( 'is_checkout' ) || ( is_checkout() && ! empty( is_wc_endpoint_url('order-received') ) ) ) return;
		
		$settings = $this->get_settings_for_display();

		/**
		 * Load attributes
		 */
		$this->render_editing_attributes();

		if( wcd_is_edit_mode() ) {
			$_wcd_checkout_fields = get_option( '_wcd_checkout_fields', [] );
			$_wcd_checkout_fields['shipping'] = $settings['shipping_form_items'];
			update_option( '_wcd_checkout_fields', $_wcd_checkout_fields );
		}

		$checked = $settings['default_checked'] == 'yes' ? 'checked' : '';
		$shipping_fields = [];
		foreach( $settings['shipping_form_items'] as $item ) {	
			$shipping_fields[ sanitize_text_field( $item['shipping_input_name'] ) ] = 
		        [
		            'label'			=> sanitize_text_field( $item['shipping_input_label'] ),
		            'type'			=> $item['shipping_input_type'],
		            'required'		=> $item['shipping_input_required'] == 'true' ? true : false,
		            'class'			=> is_array( $item['shipping_input_class'] ) ? $item['shipping_input_class'] : explode( ' ', $item['shipping_input_class'] ),
		            'autocomplete'	=> sanitize_text_field( $item['shipping_input_autocomplete'] ), 
		            'placeholder'	=> sanitize_text_field( $item['shipping_input_placeholder'] ), 
		            'priority'		=> 10,
		        ];
		        
	        if ( isset( $item['shipping_input_options'] ) ) {
	        	$options = explode( PHP_EOL, $item['shipping_input_options'] );
	        	$new_options = [];
	        	foreach ( $options as $option ) {
	        		$new_options[ strtolower( $option ) ] = $option;
	        	}
	        	$shipping_fields[ sanitize_text_field( $item['shipping_input_name'] ) ]['options'] = $new_options;
	        }
		}

		if( 'yes' == $settings['shipping_title_show'] ):

			printf( '<%1$s %2$s>%3$s</%1$s>',
				esc_attr( $settings['payment_title_tag'] ),
				$this->get_render_attribute_string( 'shipping_title_text' ),
				esc_html( $settings['shipping_title_text'] )
			);

		endif; 
		?>

		<p id="shipping-checkbox" class="shipping-checkbox-area">
			<label class="shipping-checkbox-label">
				<input id="shipping-checkbox-input" class="shipping-checkbox-input" type="checkbox" name="ship_to_different_address" value="1" <?php esc_html_e( $checked ); ?>> <span class="shipping-checkbox-caption"><?php echo esc_html( $settings['shipping_toggle_caption'] ); ?></span>
			</label>
		</p>
		<div class="wl-shipping">
			<?php

			foreach ( $shipping_fields as $key => $field ) {
				woocommerce_form_field( $key, $field, WC()->checkout->get_value( $key ) );
			}
		?>
		</div>
			
		<?php
		
		do_action( 'codesigner_after_main_content', $this );

		/**
		 * Load Script
		 */
		$this->render_script( $settings );
	}

	private function render_editing_attributes() {
		$this->add_inline_editing_attributes( 'shipping_title_text', 'none' );
		$this->add_render_attribute( 'shipping_title_text', 'class', 'wl-shipping-title' );
	}

	protected function render_script( $settings ) {
		?>
		<script type="text/javascript">
			jQuery(function($){
				<?php if ( 'yes' != $settings['default_checked'] ) :  ?> 
					$('.wl-shipping').hide()
				<?php endif; ?>
            	$('.shipping-checkbox-area').on('click',function () {
        	        if ($('.shipping-checkbox-input').is(':checked')) {
        	        	$('.wl-shipping').slideDown()
        	        }else{
        	        	$('.wl-shipping').slideUp()
        	        }
        	    });
        	})
		</script>
		<?php
	}
}

